/**
 * @module cbitmap
 * @author Manuel Mausz, 0728348
 * @brief  Implementation of CFile handling Windows Bitmaps.
 * @date   17.04.2009
 */

#ifndef CBITMAP_H
#define CBITMAP_H

#include "cfile.h"

class CPixelFormat;
#include "cpixelformat.h"

/**
 * @class CBitmap
 * @brief Implementation of CFile handling Windows Bitmaps.
 *
 * In order to support operations on bitmaps with different color bitcounts
 * different implementations of CPixelFormat are used. These classes are
 * allowed to modify the bitmap headers and pixelbuffer directly.
 *
 * On error CFile::FileError is thrown.
 */
class CBitmap : public CFile
{
  public:
    /**
     * @method CBitmap
     * @brief  Default ctor
     * @param  -
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CBitmap()
      : m_pixeldata(NULL), m_pixelformat(NULL)
    {
      m_types.insert("BMP");
    }

    /**
     * @method ~CBitmap
     * @brief  Default dtor
     * @param  -
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    ~CBitmap();

    /**
     * @method read
     * @brief  Reads Windows Bitmap from filestream.
     *         On error an exception is thrown.
     * @param  in filestream to read data from
     * @return -
     * @globalvars none
     * @exception CFile::FileError
     * @exception bad_alloc
     * @conditions none
     */
    void read(std::ifstream& in);

    /**
     * @method write
     * @brief  Writes Windows Bitmap to filestream.
     * @param  out filestream to read data from
     * @return -
     * @globalvars none
     * @exception  FileError
     * @exception  bad_alloc
     * @conditions none
     */
    void write(std::ofstream& out);

    /**
     * @method callFunc
     * @brief  Delegates the function and its parameters to the correct
     *         internal method
     * @param  func   function name
     * @param  params function parameters as list
     * @return -
     * @globalvars none
     * @exception  ParserError
     * @conditions none
     */
    void callFunc(const std::string& func, const std::list<std::string>& params);

#ifdef DEBUG
    /**
     * @method dump
     * @brief  Dumps the Windows Bitmap file headers to ostream
     * @param  out output stream
     * @return -
     * @globalvars
     * @exception
     * @conditions
     */
    void dump(std::ostream& out);
#endif

    /**
     * @brief Windows Bitmap File Header structure
     */
#pragma pack(push,1)
    typedef struct
    {
      /** the magic number used to identify the BMP file */
      uint8_t  bfType[2];
      /** the size of the BMP file in bytes */
      uint32_t bfSize;
      /** reserved */
      uint32_t bfReserved;
      /** the offset of the byte where the bitmap data can be found */
      uint32_t bfOffBits;
    } BITMAP_FILEHEADER;
#pragma pack(pop)

    /**
     * @brief Windows Bitmap Info Header structure
     */
#pragma pack(push,1)
    typedef struct
    {
      /** the size of this header (40 bytes) */
      uint32_t biSize;
      /** the bitmap width in pixels (signed integer) */
      int32_t  biWidth;
      /** the bitmap height in pixels (signed integer) */
      int32_t  biHeight;
      /** the number of color planes being used. Must be set to 1 */
      uint16_t biPlanes;
      /** the number of bits per pixel, which is the color depth of the image */
      uint16_t biBitCount;
      /** the compression method being used */
      uint32_t biCompression;
      /** the image size */
      uint32_t biSizeImage;
      /** the horizontal resolution of the image (pixel per meter) */
      int32_t  biXPelsPerMeter;
      /** the vertical resolution of the image (pixel per meter) */
      int32_t  biYPelsPerMeter;
      /** the number of colors in the color palette, or 0 to default to 2^n */
      uint32_t biClrUsed;
      /** the number of important colors used, or 0 when every color is
       * important; generally ignored. */
      uint32_t biClrImportant;
    } BITMAP_INFOHEADER;
#pragma pack(pop)

    /**
     * @method getFileHeader
     * @brief  Returns reference to fileheader structure of bitmap
     * @param  -
     * @return reference to fileheader structure
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    BITMAP_FILEHEADER &getFileHeader()
    {
      return m_fileheader;
    }

    /**
     * @method getInfoHeader
     * @brief  Returns reference to infoheader structure of bitmap
     * @param  -
     * @return reference to infoheader structure
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    BITMAP_INFOHEADER &getInfoHeader()
    {
      return m_infoheader;
    }

    /**
     * @method getPixelData
     * @brief  Returns pointer to pixelbuffer
     * @param  -
     * @return pointer to pixelbuffer
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    uint8_t *getPixelData()
    {
      return m_pixeldata;
    }

  protected:
    /**
     * @method fillrect
     * @brief  Fills rectangle in image starting on position x, y
     *         width size width, height and color red, green, blue.
     * @param  params function parameters as list
     * @return -
     * @globalvars none
     * @exception  FileError
     * @conditions none
     *
     * Scriptfile syntax: fillrect(x, y, width, height, red, green, blue)
     */
    void fillrect(std::list<std::string> params);

    /* members */
    /** fileheader */
    BITMAP_FILEHEADER m_fileheader;
    /** infoheader */
    BITMAP_INFOHEADER m_infoheader;
    /** pointer to pixelbuffer */
    uint8_t *m_pixeldata;
    /** pointer to CPixelFormat implementation */
    CPixelFormat *m_pixelformat;
};

#endif

/* vim: set et sw=2 ts=2: */
