/**
 * @module cscriptparser
 * @author Manuel Mausz, 0728348
 * @brief  class for parsing simple scriptfiles
 * @date   17.04.2009
 */

#ifndef CSCRIPTPARSER_H
#define CSCRIPTPARSER_H

#include <stdexcept>
#include <string>
#include <list>
#include <set>
#include "cfile.h"

/**
 * @class CScriptparser
 *
 * Parses a simple line based scriptfile with some limitations:
 * first function (starting a block) must be a read-command,
 * last must be a write-command (ending this block).
 *
 * read- and write-commands have hard coded parameters, number#1 being a filetype.
 * Classes handling certain filetypes must be of type CFile.
 * Custom functions will be passed to CFile::callFunc().
 *
 * On error ParserError will be thrown.
 */
class CScriptparser
{
  public:
    /**
     * @class ParserError
     * @brief Exception thrown by CScriptparser
     */
    class ParserError : public std::invalid_argument {
      public:
        /**
         * @method ParserError
         * @brief  Default exception ctor
         * @param  what message to pass along
         * @return -
         * @globalvars none
         * @exception  none
         * @conditions none
         */
        ParserError(const std::string& what)
          : std::invalid_argument(what), m_line("")
        {}

        /**
         * @method ParserError
         * @brief  Custom exception ctor
         * @param  what message to pass along
         * @param  line scriptline which is currently being parsed
         * @return -
         * @globalvars none
         * @exception  none
         * @conditions none
         */
        ParserError(const std::string& what, const std::string& line)
          : std::invalid_argument(what), m_line(line)
        {}

        /**
         * @method ~ParserError
         * @brief  Default dtor
         * @param  -
         * @return -
         * @globalvars none
         * @exception  not allowed
         * @conditions none
         */
        ~ParserError() throw()
        {}

        /**
         * @method getLine
         * @brief  returns reference to currently parsed scriptline (if set)
         * @return reference to currently parsed scriptline (maybe empty string)
         * @globalvars none
         * @exception  none
         * @conditions none
         */
        const std::string &getLine()
        {
          return m_line;
        }

      private:
        /* members*/
        std::string m_line;
    };

    /**
     * @method CScriptparser
     * @brief  Default ctor
     * @param  scriptfile filename of script to parse
     * @return -
     * @globalvars none
     * @exception  bad_alloc
     * @conditions none
     */
    CScriptparser(const std::string& scriptfile);

    /**
     * @method ~CScriptparser
     * @brief  Default dtor
     * @param  -
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    ~CScriptparser();

    /**
     * @method parse
     * @brief  Start parsing the scriptfile
     * @param  -
     * @return -
     * @globalvars none
     * @exception  ParserError
     * @conditions none
     */
    void parse();

  protected:
    /**
     * @method callFunc
     * @brief  Delegates the function and its parameters to the correct
     *         method (internal or handler)
     * @param  func       function name
     * @param  funcparams function parameters as list
     * @return -
     * @globalvars none
     * @exception  ParserError
     * @conditions none
     */
    void callFunc(const std::string& func, const std::list<std::string>& funcparams);

    /**
     * @method read
     * @brief  Handles/wrappes read-command. according to the filetype the
     *         read-method of the corresponding handler will be called inside.
     * @param  funcparams function parameters as list
     * @return -
     * @globalvars none
     * @exception  ParserError
     * @conditions none
     *
     * Scriptfile syntax: read(<FILETYPE>, <FILENAME>)
     */
    void read(std::list<std::string> funcparams);

    /**
     * @method write
     * @brief  Handles/wrappes write-command. according to the filetype the
     *         write-method of the corresponding handler will be called inside.
     * @param  funcparams function parameters as list
     * @return -
     * @globalvars none
     * @exception  ParserError
     * @conditions none
     *
     * Scriptfile syntax: write(<FILETYPE>, <FILENAME>)
     */
    void write(std::list<std::string> funcparams);

  private:
    /* members */
    std::set<CFile *> m_handlers;
    std::string m_scriptfile;
    std::string m_curline;
    CFile *m_handler;
};

#endif

/* vim: set et sw=2 ts=2: */
