/**
 * @module imgsynth
 * @author Manuel Mausz, 0728348
 * @brief  imgsynth reads a scriptfile given as commandline option
 *         and executes all known function inside.
 *         On error (e.g. unknown function) the program will terminate
 * @date   17.04.2009
 * @par Exercise
 *      1
 */

#include <iostream>
#include <boost/program_options.hpp>
#include "cscriptparser.h"

using namespace std;
namespace po = boost::program_options;

/**
 * @func   main
 * @brief  program entry point
 * @param  argc  standard parameter of main
 * @param  argv  standard parameter of main
 * @return 0 on success, not 0 otherwise
 * @globalvars none
 * @exception  none
 * @conditions none
 *
 * setup commandline options, parse them and pass scriptfile to scriptparser
 * instance. On error print error message to stderr.
 * Unknown commandline options will print a usage message.
 */
int main(int argc, char* argv[])
{
  string me(argv[0]);

  /* define commandline options */
  po::options_description desc("Allowed options");
  desc.add_options()
    ("help,h", "this help message")
    ("input,i", po::value<string>(), "input scriptfile");

  /* parse commandline options */
  po::variables_map vm;
  try
  {
    po::store(po::parse_command_line(argc, argv, desc), vm);
    po::notify(vm);
  }
  catch(po::error& ex)
  {
    cerr << "Error: " << ex.what() << endl;
  }

  /* print usage upon request or missing params */
  if (vm.count("help") || !vm.count("input"))
  {
    cout << "Usage: " << me << " -i <scriptfile>" << endl;
    cout << desc << endl;
    return 0;
  }

  CScriptparser parser(vm["input"].as<string>());
  try
  {
    parser.parse();
  }
  catch(CScriptparser::ParserError& ex)
  {
    cerr << me << ": Error while processing scriptfile: " << ex.what() << endl;
    if (!ex.getLine().empty())
      cerr << "Scriptline: '" << ex.getLine() << "'" << endl;
    return 1;
  }
  catch(exception& ex)
  {
    cerr << me << ": Unexpected exception: " << ex.what() << endl;
    return 1;
  }

  return 0;
}

/* vim: set et sw=2 ts=2: */
