/**
 * @module cbitmap
 * @author Guenther Neuwirth (0626638), Manuel Mausz (0728348)
 * @brief  Abstract implementation of CFile handling Bitmaps.
 * @date   17.04.2009
 */

#ifndef CBITMAP_H
#define CBITMAP_H

#include <stdint.h>
#include <map>
#include "cfile.h"
#include "cpixelformat.h"

/**
 * @class CBitmap
 * @brief Implementation of CFile handling Bitmaps.
 *
 * In order to support operations on bitmaps with different color bitcounts
 * different implementations of CPixelFormat are used. These classes are
 * allowed to modify the bitmap headers and pixelbuffer directly.
 *
 * On error CFile::FileError is thrown.
 */
class CBitmap : public CFile
{
  public:
    /**
     * @method CBitmap
     * @brief  Default ctor
     * @param  -
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CBitmap()
      : m_pixeldata(NULL), m_pixelformat(NULL), m_rowsize(0)
    {}


    /**
     * @method ~CBitmap
     * @brief  Default dtor
     * @param  -
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    virtual ~CBitmap();

    /**
     * @method read
     * @brief  Reads Windows Bitmap from filestream.
     *         On error an exception is thrown.
     * @param  in filestream to read data from
     * @return -
     * @globalvars none
     * @exception CFile::FileError
     * @exception bad_alloc
     * @conditions none
     */
    virtual void read(std::ifstream& in) = 0;

    /**
     * @method write
     * @brief  Writes Windows Bitmap to filestream.
     * @param  out      filestream to read data from
     * @return -
     * @globalvars none
     * @exception  FileError
     * @exception  bad_alloc
     * @conditions none
     */
    virtual void write(std::ofstream& out) = 0;

    /**
     * @method getPixelData
     * @brief  Returns pointer to pixelbuffer
     * @param  -
     * @return pointer to pixelbuffer
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    uint8_t *getPixelData()
    {
      return m_pixeldata;
    }

    /**
     * @method getColorTable
     * @brief  Returns reference to colortable
     * @param  -
     * @return reference to colortable
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    std::map<uint32_t, CPixelFormat::RGBPIXEL *>& getColorTable()
    {
      return m_colortable;
    }

    /**
     * @method getRowSize
     * @brief  Returns number of bytes of one row
     * @param  -
     * @return number of bytes of one row
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    uint32_t getRowSize()
    {
      return m_rowsize;
    }

    /**
     * @method getPixelDataSize
     * @brief  Return size of pixelbuffer
     * @param  -
     * @return size of pixelbuffer
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    virtual const uint32_t getPixelDataSize() = 0;

    /**
     * @method getHeight
     * @brief  Return height of bitmap in pixel
     * @param  -
     * @return height of bitmap in pixel
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    virtual const uint32_t getHeight() = 0;

    /**
     * @method getWidth
     * @brief  Return width of bitmap in pixel
     * @param  -
     * @return width of bitmap in pixel
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    virtual const uint32_t getWidth() = 0;

    /**
     * @method isMirrored
     * @brief  Windows Bitmaps can be stored upside down
     * @param  -
     * @return true if bitmap is stored upside down. false otherwise
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    virtual const bool isMirrored() = 0;

    /**
     * @method hasColorTable
     * @brief  Check if bitmap has a colortable
     *         (we don't support this yet for windows bitmaps)
     * @param  -
     * @return true if bitmap has a colortable. false otherwise
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    virtual const bool hasColorTable() = 0;

  protected:
    /**
     * @method callFunc
     * @brief  Delegates the function and its parameters to the correct
     *         internal method
     * @param  func   function name
     * @param  params function parameters as list
     * @return -
     * @globalvars none
     * @exception  ParserError
     * @conditions none
     */
    void callFunc(const std::string& func, const std::list<std::string>& params);

    /**
     * @method fillrect
     * @brief  Fills rectangle in image starting on position x, y
     *         width size width, height and color red, green, blue.
     * @param  params function parameters as list
     * @return -
     * @globalvars none
     * @exception  FileError
     * @conditions none
     *
     * Scriptfile syntax: fillrect(x, y, width, height, red, green, blue)
     */
    void fillrect(std::list<std::string> params);

    /**
     * @method invert
     * @brief  Invert image
     * @param  params function parameters as list
     * @return -
     * @globalvars none
     * @exception  FileError
     * @conditions none
     *
     * Scriptfile syntax: invert()
     */
    void invert(std::list<std::string> params);

    /**
     * @method brightness
     * @brief  Increase/decrease brightness of image
     * @param  params function parameters as list
     * @return -
     * @globalvars none
     * @exception  FileError
     * @conditions none
     *
     * Scriptfile syntax: brightness(factor)
     */
    void brightness(std::list<std::string> params);

    /**
     * @method mirror_y
     * @brief  Mirror image around the y-axis
     * @param  params function parameters as list
     * @return -
     * @globalvars none
     * @exception  FileError
     * @conditions none
     *
     * Scriptfile syntax: mirror_y()
     */
    void mirror_y(std::list<std::string> params);

    /**
     * @method mirror_x
     * @brief  Mirror image around the x-axis
     * @param  params function parameters as list
     * @return -
     * @globalvars none
     * @exception  FileError
     * @conditions none
     *
     * Scriptfile syntax: mirror_y()
     */
    void mirror_x(std::list<std::string> params);

    /* members */
    /** pointer to pixelbuffer */
    uint8_t *m_pixeldata;
    /** colortable map */
    std::map<uint32_t, CPixelFormat::RGBPIXEL *> m_colortable;
    /** set of supported PixelFormat handlers */
    std::set<CPixelFormat *> m_handlers;
    /** pointer to CPixelFormat implementation */
    CPixelFormat *m_pixelformat;
    /** number of bytes of one row in the image */
    uint32_t m_rowsize;
};

#endif

/* vim: set et sw=2 ts=2: */
