/**
 * @module cfile
 * @author Guenther Neuwirth (0626638), Manuel Mausz (0728348)
 * @brief  Abstract class for handling files.
 *         Needed for generic use in CScriptparser.
 * @date   17.04.2009
 */

#ifndef CFILE_H
#define CFILE_H

#include <string>
#include <set>
#include <list>
#include <fstream>
#include <stdexcept>

/**
 * @class CFile
 * @brief Abstract class for handling files. Needed for generic use in
 *        CScriptparser.
 *
 * In order for CScriptparser to determine which instance of CFile supports
 * which filetype, every implemententation need to insert their filetypes to
 * the member m_types in their constructor.
 *
 * On error throw FileError.
 */
class CFile
{
  public:
    /**
     * @class FileError
     * @brief Exception thrown by implemententations of CFile
     */
    class FileError : public std::invalid_argument {
      public:
        /**
         * @method FileError
         * @brief  Default exception ctor
         * @param  what message to pass along
         * @return -
         * @globalvars none
         * @exception  none
         * @conditions none
         */
        FileError(const std::string& what)
          : std::invalid_argument(what)
        {}
    };

    /**
     * @method ~CFile
     * @brief  Default dtor (virtual)
     * @param  -
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    virtual ~CFile()
    {};

    /**
     * @method read
     * @brief  Pure virtual method (interface). Should read data from filestream.
     * @param  in filestream to read data from
     * @return -
     * @globalvars none
     * @exception  FileError
     * @conditions none
     */
    virtual void read(std::ifstream& in) = 0;

    /**
     * @method write
     * @brief  Pure virtual method (interface). Should write data to filestream.
     * @param  out      filestream to write data to
     * @return -
     * @globalvars none
     * @exception  FileError
     * @conditions none
     */
    virtual void write(std::ofstream& out) = 0;

    /**
     * @method callFunc
     * @brief  Pure virtual method (interface). Should delegate the function
     *         and its parameters to the correct internal method.
     * @param  func   function name
     * @param  params function parameters as list
     * @return -
     * @globalvars none
     * @exception  FileError
     * @conditions none
     */
    virtual void callFunc(const std::string& func, const std::list<std::string>& params) = 0;

    /**
     * @method supportsType
     * @brief  Check if filetype is supported by this implementation.
     * @param  type filetype
     * @return true if filetype is supported. false otherwise
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    bool supportsType(const std::string& type)
    {
      return (m_types.find(type) == m_types.end()) ? false : true;
    }

  protected:
    /* members */
    /** set of filetypes suppported by this implementation */
    std::set<std::string> m_types;
};

#endif

/* vim: set et sw=2 ts=2: */
