/**
 * @module cpixelformat
 * @author Guenther Neuwirth (0626638), Manuel Mausz (0728348)
 * @brief  Abstract class for handling different color bitcount of Bitmaps.
 *         Needed for generic use in CBitmap.
 * @date   18.04.2009
 */

#ifndef CPIXELFORMAT_H
#define CPIXELFORMAT_H

#include <fstream>
#include <stdexcept>

class CBitmap;

/**
 * @class CPixelFormat
 * @brief Abstract class for handling different color bitcount of Bitmaps.
 *
 * Needed for generic use in CBitmap.
 *
 * On error throw PixelFormatError.
 */
class CPixelFormat
{
  public:
    /**
     * @class PixelFormatError
     * @brief Exception thrown by implemententations of CPixelFormat
     */
    class PixelFormatError : public std::invalid_argument {
      public:
        /**
         * @method PixelFormatError
         * @brief  Default exception ctor
         * @param  what message to pass along
         * @return -
         * @globalvars none
         * @exception  none
         * @conditions none
         */
        PixelFormatError(const std::string& what)
          : std::invalid_argument(what)
        {}
    };

    /**
     * @method CPixelFormat
     * @brief  Default ctor
     * @param  bitmap pointer to CBitmap instance
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CPixelFormat(CBitmap *bitmap)
      : m_bitmap(bitmap)
    {}

    /**
     * @method ~CPixelFormat
     * @brief  Default dtor (virtual)
     * @param  -
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    virtual ~CPixelFormat()
    {};

    /**
     * @brief RGB Pixel structure
     */
    typedef struct
    {
      /** red */
      uint32_t red;
      /** green */
      uint32_t green;
      /** blue */
      uint32_t blue;
    } RGBPIXEL;

    /**
     * @method getPixel
     * @brief  Get pixel at coordinates x, y
     * @param  pixel reference to pixel data
     * @param  x     x-coordinate
     * @param  y     y-coordinate
     * @return -
     * @globalvars none
     * @exception  PixelFormatError
     * @conditions none
     */
    virtual void getPixel(RGBPIXEL& pixel, const uint32_t x, const uint32_t y) = 0;

    /**
     * @method setPixel
     * @brief  Modifies pixel at coordinates x, y
     * @param  pixel reference to new pixel data
     * @param  x     x-coordinate
     * @param  y     y-coordinate
     * @return -
     * @globalvars none
     * @exception  PixelFormatError
     * @conditions none
     */
    virtual void setPixel(const RGBPIXEL& pixel, const uint32_t x, const uint32_t y) = 0;

    /**
     * @method getBitCount
     * @brief  returns color bitcount supported by this class
     * @param  -
     * @return color bitcount supported by this class
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    virtual uint32_t getBitCount() = 0;

    /**
     * @method getMaxColor
     * @brief  Get maximum values for RGB pixel
     * @param  pixel reference to pixel struct
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    virtual void getMaxColor(RGBPIXEL& pixel) = 0;

  protected:
    /* members */
    /** pointer to CBitmap instance */
    CBitmap *m_bitmap;
};

#endif

/* vim: set et sw=2 ts=2: */
