/**
 * @module cpixelformat_BGR24
 * @author Guenther Neuwirth (0626638), Manuel Mausz (0728348)
 * @brief  Implementation of CPixelFormat handling 24bit color Windows Bitmaps.
 * @date   18.04.2009
 */

#include <boost/numeric/conversion/cast.hpp>
#include <assert.h>
#include "cpixelformat_bgr24.h"
#include "cbitmap.h"

using namespace std;

void CPixelFormat_BGR24::getPixel(RGBPIXEL& pixel, uint32_t x, uint32_t y)
{
  if (m_bitmap->getPixelData() == NULL)
    throw PixelFormatError("No pixelbuffer allocated.");
  assert(m_bitmap->getPixelDataSize() > 0);
  assert(m_bitmap->getRowSize() > 0);

  /* if the y-coordinates are mirrored */
  if (m_bitmap->isMirrored())
    y = m_bitmap->getHeight() - y - 1;
  uint32_t offset = y * m_bitmap->getRowSize() + x * (4 * getBitCount() / 32);

  /* boundary check */
  if (offset + getBitCount()/8 > m_bitmap->getPixelDataSize())
    throw PixelFormatError("Pixel position is out of range.");

  /* get pixel */
  pixel.red   = *(m_bitmap->getPixelData() + offset + 2);
  pixel.green = *(m_bitmap->getPixelData() + offset + 1);
  pixel.blue  = *(m_bitmap->getPixelData() + offset);
}

/*----------------------------------------------------------------------------*/

void CPixelFormat_BGR24::setPixel(const RGBPIXEL& pixel, uint32_t x, uint32_t y)
{
  if (m_bitmap->getPixelData() == NULL)
    throw PixelFormatError("No pixelbuffer allocated.");
  assert(m_bitmap->getPixelDataSize() > 0);
  assert(m_bitmap->getRowSize() > 0);

  /* if the y-coordinates are mirrored */
  if (m_bitmap->isMirrored())
    y = m_bitmap->getHeight() - y - 1;
  uint32_t offset = y * m_bitmap->getRowSize() + x * (4 * getBitCount() / 32);

  /* boundary check */
  if (offset + getBitCount()/8 > m_bitmap->getPixelDataSize())
    throw PixelFormatError("Pixel position is out of range.");

  /* convert color values to correct types */
  uint8_t data[3];
  try
  {
    data[0] = boost::numeric_cast<uint8_t>(pixel.blue);
    data[1] = boost::numeric_cast<uint8_t>(pixel.green);
    data[2] = boost::numeric_cast<uint8_t>(pixel.red);
  }
  catch(boost::numeric::bad_numeric_cast& ex)
  {
    throw PixelFormatError("Unable to convert pixelcolor to correct size: " + string(ex.what()));
  }

  copy(data, data + 3, m_bitmap->getPixelData() + offset);
}

/* vim: set et sw=2 ts=2: */
