/**
 * @module cpixelformat_BGR555
 * @author Guenther Neuwirth (0626638), Manuel Mausz (0728348)
 * @brief  Implementation of CPixelFormat handling BGR555 color (real color)
 *         Windows Bitmaps.
 * @date   18.04.2009
 */

#include <boost/numeric/conversion/cast.hpp>
#include <assert.h>
#include "cpixelformat_bgr555.h"
#include "cbitmap.h"

using namespace std;

void CPixelFormat_BGR555::getPixel(RGBPIXEL& pixel, uint32_t x, uint32_t y)
{
  if (m_bitmap->getPixelData() == NULL)
    throw PixelFormatError("No pixelbuffer allocated.");
  assert(m_bitmap->getPixelDataSize() > 0);
  assert(m_bitmap->getRowSize() > 0);

  /* if the y-coordinates are mirrored */
  if (m_bitmap->isMirrored())
    y = m_bitmap->getHeight() - y - 1;
  uint32_t offset = y * m_bitmap->getRowSize() + x * (4 * getBitCount() / 32);

  /* boundary check */
  if (offset + getBitCount()/8 > m_bitmap->getPixelDataSize())
    throw PixelFormatError("Pixel position is out of range.");

  /* get pixel */
  uint8_t *o = m_bitmap->getPixelData() + offset;
  pixel.blue  = (*(uint16_t *)o & BGR555_BLUE_MASK)  >> BGR555_BLUE_SHIFT;
  pixel.green = (*(uint16_t *)o & BGR555_GREEN_MASK) >> BGR555_GREEN_SHIFT;
  pixel.red   = (*(uint16_t *)o & BGR555_RED_MASK)   >> BGR555_RED_SHIFT;
}

/*----------------------------------------------------------------------------*/

void CPixelFormat_BGR555::setPixel(const RGBPIXEL& pixel, uint32_t x, uint32_t y)
{
  if (m_bitmap->getPixelData() == NULL)
    throw PixelFormatError("No pixelbuffer allocated.");
  assert(m_bitmap->getPixelDataSize() > 0);
  assert(m_bitmap->getRowSize() > 0);

  /* if the y-coordinates are mirrored */
  if (m_bitmap->isMirrored())
    y = m_bitmap->getHeight() - y - 1;
  uint32_t offset = y * m_bitmap->getRowSize() + x * (4 * getBitCount() / 32);

  /* boundary check */
  if (offset + getBitCount()/8 > m_bitmap->getPixelDataSize())
    throw PixelFormatError("Pixel position is out of range.");

  /* convert color values to correct types */
  uint8_t *o = m_bitmap->getPixelData() + offset;
  *(uint16_t *)o = (uint16_t)(((pixel.blue << BGR555_BLUE_SHIFT)   & BGR555_BLUE_MASK)  |
                              ((pixel.green << BGR555_GREEN_SHIFT) & BGR555_GREEN_MASK) |
                              ((pixel.red << BGR555_RED_SHIFT)     & BGR555_RED_MASK));
}

/* vim: set et sw=2 ts=2: */
