/**
 * @module cpixelformat_bgr555
 * @author Guenther Neuwirth (0626638), Manuel Mausz (0728348)
 * @brief  Implementation of CPixelFormat handling BGR555 color (real color)
 *         Windows Bitmaps.
 * @date   26.04.2009
 */

#ifndef CPIXELFORMAT_BGR555_H
#define CPIXELFORMAT_BGR555_H

#include <fstream>
#include "cpixelformat.h"

/* to (un-)pack pixel values */
#define BGR555_RED_SHIFT    10
#define BGR555_GREEN_SHIFT  5
#define BGR555_BLUE_SHIFT   0
#define BGR555_RED_MASK     0x7C00
#define BGR555_GREEN_MASK   0x03E0
#define BGR555_BLUE_MASK    0x001F

/**
 * @class CPixelFormat_BGR555
 * @brief Implementation of CPixelFormat handling BGR555 color (real color)
 *        Windows Bitmaps.
 *
 * On error CPixelFormat::PixelFormatError is thrown.
 */
class CPixelFormat_BGR555 : public CPixelFormat
{
  public:
    /**
     * @method CPixelFormat_BGR55
     * @brief  Default ctor
     * @param  bitmap pointer to CBitmap instance
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CPixelFormat_BGR555(CBitmap *bitmap)
      : CPixelFormat(bitmap)
    {}

    /**
     * @method ~CPixelFormat_BGR555
     * @brief  Default dtor
     * @param  -
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    ~CPixelFormat_BGR555()
    {}

    /**
     * @method getPixel
     * @brief  Get pixel at coordinates x, y
     * @param  pixel reference to pixel data
     * @param  x     x-coordinate
     * @param  y     y-coordinate
     * @return -
     * @globalvars none
     * @exception  PixelFormatError
     * @conditions none
     */
    void getPixel(RGBPIXEL& pixel, uint32_t x, uint32_t y);

    /**
     * @method setPixel
     * @brief  Modifies pixel at coordinates x, y
     * @param  pixel reference to new pixel data
     * @param  x     x-coordinate
     * @param  y     y-coordinate
     * @return -
     * @globalvars none
     * @exception  PixelFormatError
     * @conditions none
     */
    void setPixel(const RGBPIXEL& pixel, uint32_t x, uint32_t y);

    /**
     * @method getBitCount
     * @brief  returns color bitcount supported by this class
     * @param  -
     * @return color bitcount supported by this class
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    uint32_t getBitCount()
    {
      return 16;
    }

    /**
     * @method getMaxColor
     * @brief  Get maximum values for RGB pixel
     * @param  pixel reference to pixel struct
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    void getMaxColor(RGBPIXEL& pixel)
    {
      /* value = 2^5 - 1 */
      pixel.red = pixel.green = pixel.blue = 31;
    }
};

#endif

/* vim: set et sw=2 ts=2: */
