/**
 * @module CPixelFormat_Indexed8
 * @author Guenther Neuwirth (0626638), Manuel Mausz (0728348)
 * @brief  Implementation of CPixelFormat handling 24bit indexed bitmaps.
 * @date   02.05.2009
 */

#include <boost/numeric/conversion/cast.hpp>
#include <assert.h>
#include "cpixelformat_indexed8.h"
#include "cbitmap.h"

using namespace std;

void CPixelFormat_Indexed8::getPixel(RGBPIXEL& pixel, uint32_t x, uint32_t y)
{
  if (m_bitmap->getPixelData() == NULL)
    throw PixelFormatError("No pixelbuffer allocated.");
  if (m_bitmap->getColorTable().size() == 0)
    return;
  assert(m_bitmap->getPixelDataSize() > 0);

  uint32_t offset = y * m_bitmap->getWidth() + x;

  /* boundary check */
  if (offset * sizeof(uint32_t) + sizeof(uint32_t) > m_bitmap->getPixelDataSize())
    throw PixelFormatError("Pixel position is out of range.");

  uint32_t color = *((uint32_t *)m_bitmap->getPixelData() + offset);

  map<uint32_t, RGBPIXEL *>::iterator it;
  if ((it = m_bitmap->getColorTable().find(color)) == m_bitmap->getColorTable().end())
    throw PixelFormatError("Pixel has no reference in colortable.");

  pixel.red   = (*it).second->red;
  pixel.green = (*it).second->green;
  pixel.blue  = (*it).second->blue;
}

/*----------------------------------------------------------------------------*/

void CPixelFormat_Indexed8::setPixel(const RGBPIXEL& pixel, uint32_t x, uint32_t y)
{
  if (m_bitmap->getPixelData() == NULL)
    throw PixelFormatError("No pixelbuffer allocated.");
  /* if colortable is empty there are no pixels */
  if (m_bitmap->getColorTable().size() == 0)
    return;
  assert(m_bitmap->getPixelDataSize() > 0);

  uint32_t offset = y * m_bitmap->getWidth() + x;

  /* boundary check */
  if (offset * sizeof(uint32_t) + sizeof(uint32_t) > m_bitmap->getPixelDataSize())
    throw PixelFormatError("Pixel position is out of range.");

  /* try to look up color in colortable */
  map<uint32_t, RGBPIXEL *>::iterator it;
  for(it = m_bitmap->getColorTable().begin(); it != m_bitmap->getColorTable().end(); it++)
  {
    if ((*it).second->red   == pixel.red &&
        (*it).second->green == pixel.green &&
        (*it).second->blue  == pixel.blue)
      break;
  }

  uint32_t index = (*it).first;
  /* need to get a new entry for our color */
  if (it == m_bitmap->getColorTable().end())
  {
    index = (*it).first + 1;
    RGBPIXEL *pixelptr = new RGBPIXEL;
    pixelptr->red   = pixel.red;
    pixelptr->green = pixel.green;
    pixelptr->blue  = pixel.blue;
    m_bitmap->getColorTable()[ index ] = pixelptr;
  }

  /* set color */
  *((uint32_t *)m_bitmap->getPixelData() + offset) = index;
}

/* vim: set et sw=2 ts=2: */
