/**
 * @module CPixmap
 * @author Guenther Neuwirth (0626638), Manuel Mausz (0728348)
 * @brief  Implementation of CFile CBitmap handling XPM.
 * @date   27.04.2009
 */

#ifndef CPixmap_H
#define CPixmap_H

#include <stdint.h>
#include "cbitmap.h"

#define PIXMAP_IDENTIFIER "/* XPM */"
#define PIXMAP_COLORCHARS ".#abcdefghijklmnopqrstuvwxyzABCD" \
                          "EFGHIJKLMNOPQRSTUVWXYZ0123456789"

/**
 * @class CPixmap
 * @brief Implementation of CFile handling Pixmap file format.
 *
 * In order to support operations on pixmaps in color mode an 
 * implementations of CPixelFormat is used. These classe are
 * allowed to modify the pixmap header, pixelbuffer and color table directly.
 *
 * On error CFile::FileError is thrown.
 */
class CPixmap : public CBitmap
{
  public:
    /**
     * @method CPixmap
     * @brief  Default ctor
     * @param  -
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CPixmap();

    /**
     * @method ~CPixmap
     * @brief  Default dtor
     * @param  -
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    ~CPixmap()
      {}

    /**
     * @method read
     * @brief  Reads Pixmap from filestream.
     *         On error an exception is thrown.
     * @param  in filestream to read data from
     * @return -
     * @globalvars none
     * @exception CFile::FileError
     * @exception bad_alloc
     * @conditions none
     */
    void read(std::ifstream& in);

    /**
     * @method write
     * @brief  Writes Pixmap to filestream.
     * @param  out filestream to read data from
     * @return -
     * @globalvars none
     * @exception  FileError
     * @exception  bad_alloc
     * @conditions none
     */
    void write(std::ofstream& out);

#ifdef DEBUG
    /**
     * @method dump
     * @brief  Dumps the Pixmap file header and pixel data to ostream
     * @param  out output stream
     * @return -
     * @globalvars
     * @exception
     * @conditions
     */
    void dump(std::ostream& out);
#endif

    /**
     * @method getPixelDataSize
     * @brief  Return size of pixelbuffer
     * @param  -
     * @return size of pixelbuffer
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    const uint32_t getPixelDataSize()
    {
      return m_fileheader.width * m_fileheader.height * sizeof(uint32_t);
    }

    /**
     * @method getHeight
     * @brief  Return height of bitmap in pixel
     * @param  -
     * @return height of bitmap in pixel
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    const uint32_t getHeight()
    {
      return m_fileheader.height;
    }

    /**
     * @method getWidth
     * @brief  Return width of bitmap in pixel
     * @param  -
     * @return width of bitmap in pixel
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    const uint32_t getWidth()
    {
      return m_fileheader.width;
    }

    /**
     * @method hasColorTable
     * @brief  Check if bitmap has a colortable
     *         (we don't support this yet for windows bitmaps)
     * @param  -
     * @return true if bitmap has a colortable. false otherwise
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    const bool hasColorTable()
    {
      return true;
    }

    /**
     * @method isMirrored
     * @brief  Windows Bitmaps can be stored upside down
     * @param  -
     * @return true if bitmap is stored upside down. false otherwise
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    const bool isMirrored()
    {
      /* pixmap is never mirrored */
      return false;
    }

  protected:
    /**
     * @method getLine
     * @brief  read trimmed line (terminated by \n) from filestream
     * @param  in              filestream to read data from
     * @param  ignore_comments true: ignore c-like comments
     * @return return trimmed line from filestream
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    std::string getLine(std::ifstream& in, bool ignore_comments = true);

    /**
     * @method getArrayLine
     * @brief  read trimmed c-arrayline from filestream
     * @param  in filestream to read data from
     * @return return trimmed c-arrayline from filestream
     * @globalvars none
     * @exception  FileError
     * @conditions none
     */
    std::string getCArrayLine(std::ifstream& in);

    /**
     * @method getXPMColorID
     * @brief  get xpm color identifier, generated using an index
     * @param  index  index used to generate the xpm color identifier
     * @param  length length of xpm color identifier
     * @return return xpm color identifier, generated using index
     * @globalvars none
     * @exception  FileError
     * @conditions none
     */
    const std::string getXPMColorID(unsigned int index, unsigned int length);

    /**
     * @brief Pixmap Header structure
     */
    typedef struct
    {
      /** the xpm width in pixels (signed integer) */
      uint32_t width;
      /** the xpm height in pixels (signed integer) */
      uint32_t height;
      /** the number of colors (signed integer) */
      uint32_t nColor;
      /** the number of characters per pixel (signed integer) */
      uint32_t nChar;
      /** X-Position Hotspots */
      uint32_t xHotspot;
      /** Y-Position Hotspots */
      uint32_t yHotspot;
      /** is hotspot set */
      bool _HOTSPOT;
      /** XPMEXT extension tag found*/
      bool _XPMEXT;
      /** unchanged extension */
      std::string extension;
    } PIXMAP_FILEHEADER;

    /* members */
    /** fileheader */
    PIXMAP_FILEHEADER m_fileheader;
    /** name of image/c-array */
    std::string m_imagename;
};

#endif

/* vim: set et sw=2 ts=2: */
