/**
 * @module cwindowsbitmap
 * @author Guenther Neuwirth (0626638), Manuel Mausz (0728348)
 * @brief  Implementation of CBitmap handling Windows Bitmaps.
 * @date   17.04.2009
 */

#ifndef CWINDOWSBITMAP_H
#define CWINDOWSBITMAP_H

#include <stdint.h>
#include "cbitmap.h"

/**
 * @class CWindowsBitmap
 * @brief Implementation of CBitmap handling Windows Bitmaps.
 *
 * On error CFile::FileError is thrown.
 */
class CWindowsBitmap : public CBitmap
{
  public:
    /**
     * @method CWindowsBitmap
     * @brief  Default ctor
     * @param  -
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CWindowsBitmap();

    /**
     * @method ~CWindowsBitmap
     * @brief  Default dtor
     * @param  -
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    ~CWindowsBitmap()
    {}

    /**
     * @method read
     * @brief  Reads Windows Bitmap from filestream.
     *         On error an exception is thrown.
     * @param  in filestream to read data from
     * @return -
     * @globalvars none
     * @exception CFile::FileError
     * @exception bad_alloc
     * @conditions none
     */
    void read(std::ifstream& in);

    /**
     * @method write
     * @brief  Writes Windows Bitmap to filestream.
     * @param  out      filestream to read data from
     * @return -
     * @globalvars none
     * @exception  FileError
     * @exception  bad_alloc
     * @conditions none
     */
    void write(std::ofstream& out);

#ifdef DEBUG
    /**
     * @method dump
     * @brief  Dumps the Windows Bitmap file headers to ostream
     * @param  out output stream
     * @return -
     * @globalvars
     * @exception
     * @conditions
     */
    void dump(std::ostream& out);
#endif

    /**
     * @method getPixelDataSize
     * @brief  Return size of pixelbuffer
     * @param  -
     * @return size of pixelbuffer
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    const uint32_t getPixelDataSize()
    {
      return m_infoheader.biSizeImage;
    }

    /**
     * @method getHeight
     * @brief  Return height of bitmap in pixel
     * @param  -
     * @return height of bitmap in pixel
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    const uint32_t getHeight()
    {
      /* width and height can be negativ */
      return static_cast<uint32_t>(abs(m_infoheader.biHeight));
    }

    /**
     * @method getWidth
     * @brief  Return width of bitmap in pixel
     * @param  -
     * @return width of bitmap in pixel
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    const uint32_t getWidth()
    {
      /* width and height can be negativ */
      return static_cast<uint32_t>(abs(m_infoheader.biWidth));
    }

    /**
     * @method isMirrored
     * @brief  Windows Bitmaps can be stored upside down
     * @param  -
     * @return true if bitmap is stored upside down. false otherwise
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    const bool isMirrored()
    {
      /* if height is positive the y-coordinates are mirrored */
      return (m_infoheader.biHeight > 0) ? true : false;
    }

    /**
     * @method hasColorTable
     * @brief  Check if bitmap has a colortable
     *         (we don't support this yet for windows bitmaps)
     * @param  -
     * @return true if bitmap has a colortable. false otherwise
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    const bool hasColorTable()
    {
      return false;
    }

  protected:
    /**
     * @brief Windows Bitmap File Header structure
     */
#pragma pack(push,1)
    typedef struct
    {
      /** the magic number used to identify the BMP file */
      uint8_t  bfType[2];
      /** the size of the BMP file in bytes */
      uint32_t bfSize;
      /** reserved */
      uint32_t bfReserved;
      /** the offset of the byte where the bitmap data can be found */
      uint32_t bfOffBits;
    } BITMAP_FILEHEADER;
#pragma pack(pop)

    /**
     * @brief Windows Bitmap Info Header structure
     */
#pragma pack(push,1)
    typedef struct
    {
      /** the size of this header (40 bytes) */
      uint32_t biSize;
      /** the bitmap width in pixels (signed integer) */
      int32_t  biWidth;
      /** the bitmap height in pixels (signed integer) */
      int32_t  biHeight;
      /** the number of color planes being used. Must be set to 1 */
      uint16_t biPlanes;
      /** the number of bits per pixel, which is the color depth of the image */
      uint16_t biBitCount;
      /** the compression method being used */
      uint32_t biCompression;
      /** the image size */
      uint32_t biSizeImage;
      /** the horizontal resolution of the image (pixel per meter) */
      int32_t  biXPelsPerMeter;
      /** the vertical resolution of the image (pixel per meter) */
      int32_t  biYPelsPerMeter;
      /** the number of colors in the color palette, or 0 to default to 2^n */
      uint32_t biClrUsed;
      /** the number of important colors used, or 0 when every color is
       * important; generally ignored. */
      uint32_t biClrImportant;
    } BITMAP_INFOHEADER;
#pragma pack(pop)

    /* members */
    /** fileheader */
    BITMAP_FILEHEADER m_fileheader;
    /** infoheader */
    BITMAP_INFOHEADER m_infoheader;
};

#endif

/* vim: set et sw=2 ts=2: */
