/**
 * @module ccpu
 * @author Guenther Neuwirth (0626638), Manuel Mausz (0728348)
 * @brief  CPU implementation. Used as a container for memory and instructions.
 *         Implements an run method to execute the program (= the instructions).
 * @date   10.05.2009
 */

#ifdef DEBUG
# include <iostream>
# include <iomanip>
#endif
#include "ccpu.h"
#include "displays.h"

using namespace std;

CCPU::CCPU(const unsigned cnt)
  : m_regcnt(cnt), m_memory(NULL), m_program(NULL), m_flagzero(false), m_flagsign(false)
{
  /* create registers */
  m_registers = new CDat[cnt];
  for(unsigned i = 0; i < cnt; ++i)
    m_registers[i] = 0;

  /* create displays */
  m_displays.insert(new CDisplayWDEZ);
  m_displays.insert(new CDisplayWHEX);
}

/*----------------------------------------------------------------------------*/

CCPU::~CCPU()
{
  /* delete registers */
  delete[] m_registers;
  m_registers = NULL;

  /* delete displays */
  std::set<CDisplay *>::iterator it;
  for (it = m_displays.begin() ; it != m_displays.end(); ++it)
    delete *it;
}

/*----------------------------------------------------------------------------*/

void CCPU::run()
{
  if (m_memory == NULL)
    throw runtime_error("CPU has no memory");
  if (m_program == NULL)
    throw runtime_error("CPU has no program to execute");
  if (m_regcnt == 0)
    throw runtime_error("CPU has no registers");

  bool run = true;
  while(run)
  {
    unsigned pc = static_cast<unsigned>(m_registers[0]);

    /* end of the program reached */
    if (pc == m_program->size())
      break;

    /* pc is out of bound */
    if (pc > m_program->size())
      throw runtime_error("Programcounter is out of bound");

    /* execute instruction */
    (*m_program->at(pc))(this);
    ++m_registers[0];
  }
}

/*----------------------------------------------------------------------------*/

#if DEBUG
void CCPU::dumpRegisters(std::ostream& out)
{
  out << "[REGISTER DUMP]" << endl;
  for(unsigned i = 0; i < getRegisterCount(); ++i)
  {
    out << "[" << std::setw(4) << std::setfill('0') << i << "]  "
        << m_registers[i] << endl;
  }
}
#endif

/* vim: set et sw=2 ts=2: */
