/**
 * @module ccpu
 * @author Guenther Neuwirth (0626638), Manuel Mausz (0728348)
 * @brief  CPU implementation. Used as a container for memory and instructions.
 *         Implements a run method to execute the program (= the instructions).
 * @date   10.05.2009
 */

#ifndef CCPU_H
#define CCPU_H 1

#include <iostream>
#include <set>
#include "cdat.h"
#include "cmem.h"
#include "cprogram.h"
#include "cdisplay.h"

/**
 * @class CCPU
 *
 * CPU implementation. Used as a container for memory and instructions.
 * Implements a run method to execute the program (= the instructions).
 */
class CCPU
{
  public:
    /**
     * @method CCPU
     * @brief  Default ctor
     * @param  cnt  number of registers to allocate for this cpu
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CCPU(const unsigned cnt);

    /**
     * @method ~CCPU
     * @brief  Default dtor
     * @param  -
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    ~CCPU();

    /**
     * @method getRegisterCount
     * @brief  get number of registers
     * @param  -
     * @return number of registers
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    const unsigned getRegisterCount() const
    {
      return m_regcnt;
    }

    /**
     * @method getRegisters
     * @brief  get pointer to registers array
     * @param  -
     * @return pointer to registers array
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CDat *getRegisters() const
    {
      return m_registers;
    }

    /**
     * @method setMemory
     * @brief  set memory of cpu
     * @param  memory  pointer to memory
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    void setMemory(CMem *memory)
    {
      m_memory = memory;
    }

    /**
     * @method getMemory
     * @brief  get pointer to memory
     * @param  -
     * @return pointer to memory
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CMem *getMemory() const
    {
      return m_memory;
    }

    /**
     * @method setProgram
     * @brief  set program to execute
     * @param  program  pointer to program
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    void setProgram(const CProgram *program)
    {
      m_program = program;
    }

    /**
     * @method getProgram
     * @brief  get pointer to program
     * @param  -
     * @return pointer to program
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    const CProgram *getProgram()
    {
      return m_program;
    }

    /**
     * @method getDisplays
     * @brief  get set of pointers to displays
     * @param  -
     * @return reference to set of pointers to displays
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    const std::set<CDisplay *>& getDisplays()
    {
      return m_displays;
    }

    /**
     * @method setFlagZero
     * @brief  set zero flag
     * @param  value  new value of zero flag
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    void setFlagZero(const bool value)
    {
      m_flagzero = value;
    }

    /**
     * @method getFlagZero
     * @brief  get value of zero flag
     * @param  -
     * @return value of zero flag
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    const bool getFlagZero()
    {
      return m_flagzero;
    }

    /**
     * @method setFlagSign
     * @brief  set sign flag
     * @param  value  new value of sign flag
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    void setFlagSign(const bool value)
    {
      m_flagsign = value;
    }

    /**
     * @method getFlagSign
     * @brief  get value of sign flag
     * @param  -
     * @return value of sign flag
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    const bool getFlagSign()
    {
      return m_flagsign;
    }

    /**
     * @method run
     * @brief  execute current program
     * @param  -
     * @return -
     * @globalvars none
     * @exception  std::runtime_error
     * @conditions none
     */
    void run();

#if DEBUG
    /**
     * @method dumpRegisters
     * @brief  dump content of registers to outputstream
     * @param  out  outputstream to write to
     * @return void
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    void dumpRegisters(std::ostream& out);
#endif

  private:
    /* members */
    CDat *m_registers;
    unsigned m_regcnt;
    CMem *m_memory;
    const CProgram *m_program;
    std::set<CDisplay *> m_displays;
    bool m_flagzero;
    bool m_flagsign;
};

#endif

/* vim: set et sw=2 ts=2: */
