/**
 * @module cdat
 * @author Guenther Neuwirth (0626638), Manuel Mausz (0728348)
 * @brief  Datatype template and datatype definition for CCPU and CMem
 * @date   10.05.2009
 */

#ifndef CDAT_H
#define CDAT_H 1

#include <boost/operators.hpp>
#include <iostream>

/**
 * @class CDatT
 *
 * Datatype template for CCPU and CMem.
 */
template <class T>
class CDatT
  : boost::operators<CDatT<T> >
{
  public:
    /**
     * @method CDatT
     * @brief  Default ctor
     * @param  -
     * @return -
     * @globalvars none
     * @exception  bad_alloc
     * @conditions none
     */
    CDatT()
    {}

    /**
     * @method ~CDatT
     * @brief  Default dtor
     * @param  -
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    ~CDatT()
    {}

    /**
     * @method CDatT
     * @brief  Copy constructor for CDatT
     * @param  other  reference to CDatT which will be copied
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CDatT(const CDatT& other)
      : m_value(other.m_value)
    {}

    /**
     * @method CDatT
     * @brief  Copy constructor for int
     * @param  newval  new value for CDatT
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CDatT(T newval)
      : m_value(newval)
    {}

    /**
     * @method getValue
     * @brief  returns value of CDatT
     * @param  -
     * @return value of CDatT
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    T getValue() const
    {
      return m_value;
    }

    /**
     * @method operator T
     * @brief  convert to T
     * @param  -
     * @return T
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    operator T()
    {
      return m_value;
    }

    /**
     * @method operator<
     * @brief  implementation of operator <
     * @param  x  reference to CDatT
     * @return true if cdat is less than object x
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    bool operator<(const CDatT& x) const
    {
      return m_value < x.m_value;
    }

    /**
     * @method operator==
     * @brief  implementation of operator ==
     * @param  x  reference to CDatT
     * @return true if cdat equals object x
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    bool operator==(const CDatT& x) const
    {
      return m_value == x.m_value;
    }

    /**
     * @method operator+=
     * @brief  implementation of operator +=
     * @param  x  reference to CDatT
     * @return refecence to CDatT
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CDatT& operator+=(const CDatT& x)
    {
      m_value += x.m_value;
      return *this;
    }

    /**
     * @method operator-=
     * @brief  implementation of operator -=
     * @param  x  reference to CDatT
     * @return refecence to CDatT
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CDatT& operator-=(const CDatT& x)
    {
      m_value -= x.m_value;
      return *this;
    }

    /**
     * @method operator*=
     * @brief  implementation of operator *=
     * @param  x  reference to CDatT
     * @return refecence to CDatT
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CDatT& operator*=(const CDatT& x)
    {
      m_value *= x.m_value;
      return *this;
    }

    /**
     * @method operator/=
     * @brief  implementation of operator /=
     * @param  x  reference to CDatT
     * @return refecence to CDatT
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CDatT& operator/=(const CDatT& x)
    {
      m_value /= x.m_value;
      return *this;
    }

    /**
     * @method operator%=
     * @brief  implementation of operator %=
     * @param  x  reference to CDatT
     * @return refecence to CDatT
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CDatT& operator%=(const CDatT& x)
    {
      m_value %= x.m_value;
      return *this;
    }

    /**
     * @method operator|=
     * @brief  implementation of operator |=
     * @param  x  reference to CDatT
     * @return refecence to CDatT
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CDatT& operator|=(const CDatT& x)
    {
      m_value |= x.m_value;
      return *this;
    }

    /**
     * @method operator&=
     * @brief  implementation of operator &=
     * @param  x  reference to CDatT
     * @return refecence to CDatT
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CDatT& operator&=(const CDatT& x)
    {
      m_value &= x.m_value;
      return *this;
    }

    /**
     * @method operator^=
     * @brief  implementation of operator ^=
     * @param  x  reference to CDatT
     * @return refecence to CDatT
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CDatT& operator^=(const CDatT& x)
    {
      m_value ^= x.m_value;
      return *this;
    }

    /**
     * @method operator++
     * @brief  implementation of operator ++
     * @param  -
     * @return refecence to CDatT
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CDatT& operator++()
    {
      m_value++;
      return *this;
    }

    /**
     * @method operator--
     * @brief  implementation of operator --
     * @param  -
     * @return refecence to CDatT
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CDatT& operator--()
    {
      m_value--;
      return *this;
    }

    /**
     * @method operator<<
     * @brief  Shift/output operator for outputstream
     * @param  stream  reference to outputstream
     * @param  cdat    object which will be printed to stream
     * @return reference to outputstream
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    friend std::ostream& operator<<(std::ostream& stream, CDatT cdat)
    {
      stream << cdat.m_value;
      return stream;
    }

    /**
     * @method operator>>
     * @brief  Shift/read operator for inputstream
     * @param  stream  reference to inputstream
     * @param  cdat    reference to object which will be read from stream
     * @return reference to inputstream
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    friend std::istream& operator>>(std::istream & stream, CDatT& cdat)
    {
      stream >> cdat.m_value;
      return stream;
    }

  private:
    /* members */
    T m_value;
};

/**
 * @class CDat
 *
 * Datatype for CCPU and CMem
 */
typedef CDatT<int> CDat;

#endif

/* vim: set et sw=2 ts=2: */
