/**
 * @module cinstruction
 * @author Guenther Neuwirth (0626638), Manuel Mausz (0728348)
 * @brief  Abstract class for displays
 * @date   13.05.2009
 */

#ifndef CINSTRUCTION_H
#define CINSTRUCTION_H 1

#include <iostream>
#include <list>

/* forward declare CCPU */
class CCPU;

/**
 * @class CInstruction
 *
 * Abstract class for displays
 */
class CInstruction
{
  public:
    /**
     * @method CInstruction
     * @brief  Default ctor
     * @param  name  name of instruction
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    CInstruction(std::string name)
      : m_name(name)
    {}

    /**
     * @method ~CInstruction
     * @brief  Default dtor
     * @param  -
     * @return -
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    virtual ~CInstruction()
    {}

    /**
     * @method operator==
     * @brief  implementation of operator ==
     * @param  name reference to std::string
     * @return true if instructionname is name
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    virtual bool operator==(std::string& name)
    {
      return name == m_name;
    }

    /**
     * @method operator()
     * @brief  implementation of operator (CCPU)
     * @param  cpu  pointer to cpu
     * @return -
     * @globalvars none
     * @exception  std::runtime_error
     * @conditions none
     */
    virtual CInstruction& operator()(CCPU *cpu)
    {
      execute(cpu);
      return *this;
    }

    /**
     * @method getName
     * @brief  returns instruction name
     * @param  -
     * @return name of instruction
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    virtual const std::string& getName()
    {
      return m_name;
    }

    /**
     * @method dump
     * @brief  dumps information about instruction to outputstream
     * @param  stream  outputstream
     * @return reference to outputstream
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    virtual std::ostream& dump(std::ostream& stream)
    {
      stream << m_name;
      return stream;
    }

    /**
     * @method operator<<
     * @brief  Shift/output operator for outputstream
     * @param  stream  reference to outputstream
     * @param  instr   object which will be printed to stream
     * @return reference to outputstream
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    friend std::ostream& operator<<(std::ostream& stream, CInstruction& instr)
    {
      return instr.dump(stream);
    }

    /**
     * @method parseRegister
     * @brief  parses register syntax Rx (e.g. "R1")
     * @param  str  register in assembler syntax
     * @return registernumber
     * @globalvars none
     * @exception  std::runtime_error
     * @conditions none
     */
    virtual const unsigned parseRegister(const std::string& str);

    /**
     * @method checkRegister
     * @brief  performs a register boundary check
     *         does the register exist in cpu?
     * @param  cpu     pointer to cpu
     * @param  regidx  registernumber
     * @return -
     * @globalvars none
     * @exception  std::runtime_error
     * @conditions none
     */
    virtual void checkRegister(CCPU *cpu, const unsigned regidx);

    /**
     * @method factory
     * @brief  creates a new instance of this instruction
     * @param  -
     * @return new instruction instance
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    virtual CInstruction *factory() = 0;

    /**
     * @method compile
     * @brief  parses instruction parameters and prepares the
     *         instruction for executing
     * @param  params  list of parameters of this instruction
     * @return -
     * @globalvars none
     * @exception  std::runtime_error
     * @conditions none
     */
    virtual void compile(std::list<std::string>& params) = 0;

    /**
     * @method execute
     * @brief  executes the instruction
     * @param  cpu  pointer to cpu
     * @return -
     * @globalvars none
     * @exception  std::runtime_error
     * @conditions none
     */
    virtual void execute(CCPU *cpu) = 0;

  protected:
    /* members */
    /** name of instruction */
    std::string m_name;
};

#endif

/* vim: set et sw=2 ts=2: */
