/**
 * @module cmem
 * @author Guenther Neuwirth (0626638), Manuel Mausz (0728348)
 * @brief  Memory template and memory definition for CCPU
 * @date   10.05.2009
 */

#ifndef CMEM_H
#define CMEM_H 1

#include <vector>
#include <istream>
#include <sstream>
#include <stdexcept>
#include <boost/lexical_cast.hpp>
#ifdef DEBUG
# include <iostream>
# include <iomanip>
#endif
#include "cdat.h"

/**
 * @class CVectorMem
 *
 * Extends std::vector template for use as memory for CCPU.
 */
template <class T, class Allocator=std::allocator<T> >
class CVectorMem
  : public std::vector<T, Allocator>
{
  public:
    using std::vector<T, Allocator>::size;
    using std::vector<T, Allocator>::at;

    /**
     * @method initialize
     * @brief  initialize the vector with the content of istream. istream is
     *         read per line. empty lines will add unitialized elements.
     * @param  in  inputstream to read from
     * @return void
     * @globalvars none
     * @exception  std::runtime_error
     * @conditions none
     */
    void initialize(std::istream& in)
    {
      if (!in.good())
        return;

      std::string line;
      unsigned i = 0;
      while (!in.eof() && in.good())
      {
        ++i;
        std::getline(in, line);

        /* skip last line if it's empty */
        if (line.empty() && in.eof())
          break;

        T value;
        try
        {
          if (!line.empty())
            value = boost::lexical_cast<T>(line);
        }
        catch(boost::bad_lexical_cast& ex)
        {
          std::stringstream sstr;
          sstr << "Unable to convert input (line " << i << "): " << ex.what();
          throw std::runtime_error(sstr.str());
        }

        push_back(value);
      }
    }

#if DEBUG
    /**
     * @method dump
     * @brief  dumps contents of vector to outputstream
     * @param  out  outputstream to write to
     * @return void
     * @globalvars none
     * @exception  none
     * @conditions none
     */
    void dump(std::ostream& out)
    {
      out << "[MEMORY DUMP]" << std::endl;
      for(unsigned i = 0; i < size(); ++i)
      {
        out << "[" << std::setw(4) << std::setfill('0') << i << "]  "
            << at(i) << std::endl;
      }
    }
#endif
};

/**
 * @class CMem
 *
 * Memory definition for CCPU
 */
typedef CVectorMem<CDat> CMem;

#endif

/* vim: set et sw=2 ts=2: */
