/**
 * @module instructions
 * @author Guenther Neuwirth (0626638), Manuel Mausz (0728348)
 * @brief  Implementations of CInstruction
 * @date   10.05.2009
 */

#ifndef INSTRUCTIONS_H
#define INSTRUCTIONS_H 1

#include "cinstruction.h"
#include "ccpu.h"

/**
 * @class CInstructionInc
 *
 * Implementation of assembler command "inc"
 * Syntax: inc R1
 *   (R1++)
 */
class CInstructionInc
 : public CInstruction
{
  public:
    CInstructionInc()
      : CInstruction("inc")
    {}

    CInstructionInc *factory()
    {
      return new CInstructionInc;
    }

    void compile(std::list<std::string>& params);
    void execute(CCPU *cpu);

  protected:
    /** register number */
    unsigned m_regidx1;
};

/*============================================================================*/

/**
 * @class CInstructionDec
 *
 * Implementation of assembler command "dec"
 * Syntax: dec R1
 *   (R1--)
 */
class CInstructionDec
 : public CInstruction
{
  public:
    CInstructionDec()
      : CInstruction("dec")
    {}

    CInstructionDec *factory()
    {
      return new CInstructionDec;
    }

    void compile(std::list<std::string>& params);
    void execute(CCPU *cpu);

  protected:
    /** register number */
    unsigned m_regidx1;
};

/*============================================================================*/

/**
 * @class CInstructionAdd
 *
 * Implementation of assembler command "add"
 * Syntax: add R1, R2, R3
 *   (R1 = R2 + R3)
 */
class CInstructionAdd
 : public CInstruction
{
  public:
    CInstructionAdd()
      : CInstruction("add")
    {}

    CInstructionAdd *factory()
    {
      return new CInstructionAdd;
    }

    void compile(std::list<std::string>& params);
    void execute(CCPU *cpu);

  protected:
    /** register number */
    unsigned m_regidx1;
    /** register number */
    unsigned m_regidx2;
    /** register number */
    unsigned m_regidx3;
};

/*============================================================================*/

/**
 * @class CInstructionSub
 *
 * Implementation of assembler command "sub"
 * Syntax: sub R1, R2, R3
 *   (R1 = R2 - R3)
 */
class CInstructionSub
 : public CInstruction
{
  public:
    CInstructionSub()
      : CInstruction("sub")
    {}

    CInstructionSub *factory()
    {
      return new CInstructionSub;
    }

    void compile(std::list<std::string>& params);
    void execute(CCPU *cpu);

  protected:
    /** register number */
    unsigned m_regidx1;
    /** register number */
    unsigned m_regidx2;
    /** register number */
    unsigned m_regidx3;
};

/*============================================================================*/

/**
 * @class CInstructionMul
 *
 * Implementation of assembler command "mul"
 * Syntax: mul R1, R2, R3
 *   (R1 = R2 * R3)
 */
class CInstructionMul
 : public CInstruction
{
  public:
    CInstructionMul()
      : CInstruction("mul")
    {}

    CInstructionMul *factory()
    {
      return new CInstructionMul;
    }

    void compile(std::list<std::string>& params);
    void execute(CCPU *cpu);

  protected:
    /** register number */
    unsigned m_regidx1;
    /** register number */
    unsigned m_regidx2;
    /** register number */
    unsigned m_regidx3;
};

/*============================================================================*/

/**
 * @class CInstructionDiv
 *
 * Implementation of assembler command "div"
 * Syntax: div R1, R2, R3
 *   (R1 = R2 / R3)
 */
class CInstructionDiv
 : public CInstruction
{
  public:
    CInstructionDiv()
      : CInstruction("div")
    {}

    CInstructionDiv *factory()
    {
      return new CInstructionDiv;
    }

    void compile(std::list<std::string>& params);
    void execute(CCPU *cpu);

  protected:
    /** register number */
    unsigned m_regidx1;
    /** register number */
    unsigned m_regidx2;
    /** register number */
    unsigned m_regidx3;
};

/*============================================================================*/

/**
 * @class CInstructionLoad
 *
 * Implementation of assembler command "load"
 * Syntax: load R1, R2
 *   (R1 = memory[R2])
 */
class CInstructionLoad
 : public CInstruction
{
  public:
    CInstructionLoad()
      : CInstruction("load")
    {}

    CInstructionLoad *factory()
    {
      return new CInstructionLoad;
    }

    void compile(std::list<std::string>& params);
    void execute(CCPU *cpu);

  protected:
    /** register number */
    unsigned m_regidx1;
    /** register number */
    unsigned m_regidx2;
};

/*============================================================================*/

/**
 * @class CInstructionStore
 *
 * Implementation of assembler command "store"
 * Syntax: store R1, R2
 *   (memory[R2] = R1)
 */
class CInstructionStore
 : public CInstruction
{
  public:
    CInstructionStore()
      : CInstruction("store")
    {}

    CInstructionStore *factory()
    {
      return new CInstructionStore;
    }

    void compile(std::list<std::string>& params);
    void execute(CCPU *cpu);

  protected:
    /** register number */
    unsigned m_regidx1;
    /** register number */
    unsigned m_regidx2;
};

/*============================================================================*/

/**
 * @class CInstructionTest
 *
 * Implementation of assembler command "test"
 * Syntax: test R1
 *   (R1 == 0: zeroflag: true, R1 < 0: signflag: true)
 */
class CInstructionTest
 : public CInstruction
{
  public:
    CInstructionTest()
      : CInstruction("test")
    {}

    CInstructionTest *factory()
    {
      return new CInstructionTest;
    }

    void compile(std::list<std::string>& params);
    void execute(CCPU *cpu);

  protected:
    /** register number */
    unsigned m_regidx1;
};

/*============================================================================*/

/**
 * @class CInstructionLabel
 *
 * Implementation of assembler command "label"
 * Syntax: label name:
 */
class CInstructionLabel
 : public CInstruction
{
  public:
    CInstructionLabel()
      : CInstruction("label")
    {}

    CInstructionLabel *factory()
    {
      return new CInstructionLabel;
    }

    void compile(std::list<std::string>& params)
    {}

    void execute(CCPU *cpu)
    {}
};

/*============================================================================*/

/**
 * @class CInstructionJumpA
 *
 * Implementation of assembler command "jumpa"
 * Syntax: jumpa labelname
 *   (jump to labelname)
 */
class CInstructionJumpA
 : public CInstruction
{
  public:
    CInstructionJumpA()
      : CInstruction("jumpa"), m_addr("")
    {}

    CInstructionJumpA *factory()
    {
      return new CInstructionJumpA;
    }

    void compile(std::list<std::string>& params);
    void execute(CCPU *cpu);

  protected:
    /** labelname */
    std::string m_addr;
};

/*============================================================================*/

/**
 * @class CInstructionJumpZ
 *
 * Implementation of assembler command "jumpz"
 * Syntax: jumpz labelname
 *   (jump to labelname if zeroflag)
 */
class CInstructionJumpZ
 : public CInstruction
{
  public:
    CInstructionJumpZ()
      : CInstruction("jumpz"), m_addr("")
    {}

    CInstructionJumpZ *factory()
    {
      return new CInstructionJumpZ;
    }

    void compile(std::list<std::string>& params);
    void execute(CCPU *cpu);

  protected:
    /** labelname */
    std::string m_addr;
};

/*============================================================================*/

/**
 * @class CInstructionJumpS
 *
 * Implementation of assembler command "jumps"
 * Syntax: jumps labelname
 *   (jump to labelname if signflag)
 */
class CInstructionJumpS
 : public CInstruction
{
  public:
    CInstructionJumpS()
      : CInstruction("jumps"), m_addr("")
    {}

    CInstructionJumpS *factory()
    {
      return new CInstructionJumpS;
    }

    void compile(std::list<std::string>& params);
    void execute(CCPU *cpu);

  protected:
    /** labelname */
    std::string m_addr;
};

/*============================================================================*/

/**
 * @class CInstructionWrite
 *
 * Implementation of assembler command "write"
 * Syntax: write DEV, R1
 *   (write R1 to DEV, which is a name of a display)
 */
class CInstructionWrite
 : public CInstruction
{
  public:
    CInstructionWrite()
      : CInstruction("write"), m_dev("")
    {}

    CInstructionWrite *factory()
    {
      return new CInstructionWrite;
    }

    void compile(std::list<std::string>& params);
    void execute(CCPU *cpu);

  protected:
    /** register number */
    unsigned m_regidx1;
    /** device name */
    std::string m_dev;
};

#endif

/* vim: set et sw=2 ts=2: */
