/*
 * Name:         global
 * Author:       Manuel Mausz, 0728348
 * Description:  Shared definitions and functions
 * Created:      24.04.2009
 */

#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include "global.h"

/* global variables */
volatile sig_atomic_t error = 0;  /* error flag */
char *me = NULL;                  /* name of myself (argv[0]) */

/* function declerations */
void free_resources(void);

/*
 * NAME: vprintferr
 * PURPOSE:
 * prints error message to stderr and sets global error flag to 1. if errno
 * is set the error message string corresponding to errno will be appended
 * output format: "argv[0]: fmt\n" respectively "argv[0]: fmt: errorstr\n"
 *
 * PARAMETERS:
 * char *fmt  ... format string to be passed to printf
 * va_list ap ... variable argument list structure
 *
 * RETURN VALUE:
 * void
 *
 * GLOBAL VARS:
 * me, error
 */
void vprintferr(const char *fmt, va_list ap)
{
  (void) fprintf(stderr, "%s: ", me);
  (void) vfprintf(stderr, fmt, ap);
  if (errno)
    (void) fprintf(stderr, ": %s", strerror(errno));
  (void) fprintf(stderr, "\n");
  error = 1;
}

/*
 * NAME: printferr
 * PURPOSE:
 * wrapper for vprintferr to support variable argument list (like printf)
 * see vprintferr for more information
 *
 * PARAMETERS:
 * char *fmt ... format string to be passed to vprintferr (see printf)
 * ...       ... arguments according to fmt (see printf)
 *
 * RETURN VALUE:
 * void
 */
void printferr(const char *fmt, ...)
{
  va_list ap;

  (void) va_start(ap, fmt);
  (void) vprintferr(fmt, ap);
  (void) va_end(ap);
}

/*
 * NAME: bailout
 * PURPOSE:
 * prints formated error message using vprintferr and exits with EXIT_FAILURE
 * printing can be omitted by setting fmt to NULL
 * see vprintferr for more information
 *
 * PARAMETERS:
 * char *fmt ... format string to be passed to vprintferr (see printf)
 * ...       ... arguments according to fmt (see printf)
 *
 * RETURN VALUE:
 * void
 */
void bailout(const char *fmt, ...)
{
  va_list ap;

  if (fmt != NULL)
  {
    (void) va_start(ap, fmt);
    (void) vprintferr(fmt, ap);
    (void) va_end(ap);
  }
  (void) free_resources();
  (void) exit(EXIT_FAILURE);
}
