/*
 * Name:         error
 * Author:       Manuel Mausz, 0728348
 * Description:  Shared definitions and functions
 * Created:      24.05.2009
 */

#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include "error.h"

/* error variables */
int error = 0;    /* error flag */
char *me = NULL;  /* name of myself (argv[0]) */
int child = 0;    /* flag being the child */

/* function declerations */
void free_resources(void);

/*
 * NAME: vprintferr
 * PURPOSE:
 * prints error message to stderr and sets global error flag to 1. if perrno
 * and errno is set the error message string corresponding to errno will be
 * appended
 * output format: "argv[0]: fmt\n" respectively "argv[0]: fmt: errorstr\n"
 *
 * PARAMETERS:
 * int perrno ... print errno if errno is set
 * char *fmt  ... format string to be passed to printf
 * va_list ap ... variable argument list structure
 *
 * RETURN VALUE:
 * void
 *
 * GLOBAL VARS:
 * me, error, child
 */
void vprintferr(int perrno, const char *fmt, va_list ap)
{
  (void) fprintf(stderr, "%s: ", me);
#ifdef DEBUG
  if (child)
    (void) fprintf(stderr, "[CHILD] ");
#endif
  (void) vfprintf(stderr, fmt, ap);
  if (perrno && errno)
    (void) fprintf(stderr, ": %s", strerror(errno));
  (void) fprintf(stderr, "\n");
  error = 1;
}

/*
 * NAME: printferr
 * PURPOSE:
 * wrapper for vprintferr to support variable argument list (like printf)
 * see vprintferr for more information
 *
 * PARAMETERS:
 * int perrno ... print errno if errno is set
 * char *fmt  ... format string to be passed to vprintferr (see printf)
 * ...        ... arguments according to fmt (see printf)
 *
 * RETURN VALUE:
 * void
 */
void printferr(int perrno, const char *fmt, ...)
{
  va_list ap;

  (void) va_start(ap, fmt);
  (void) vprintferr(perrno, fmt, ap);
  (void) va_end(ap);
}

/*
 * NAME: bailout
 * PURPOSE:
 * prints formated error message using vprintferr, frees resources and exits
 * with EXIT_FAILURE. Also signal parent using SIGPIPE if its the child.
 * printing can be omitted by setting fmt to NULL
 * see vprintferr for more information
 *
 * PARAMETERS:
 * int perrno ... print errno if errno is set
 * char *fmt  ... format string to be passed to vprintferr (see printf)
 * ...        ... arguments according to fmt (see printf)
 *
 * RETURN VALUE:
 * void
 *
 * GLOBAL VARS:
 * child
 */
void bailout(int perrno, const char *fmt, ...)
{
  va_list ap;

  if (fmt != NULL)
  {
    (void) va_start(ap, fmt);
    (void) vprintferr(perrno, fmt, ap);
    (void) va_end(ap);
  }
  if (child)
    kill(getppid(), SIGPIPE);
  (void) free_resources();
  (void) exit(EXIT_FAILURE);
}
